/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */
///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Includes Section
///////////////////////////////////////////////////////////////////////////////////////////////////
#include "fsl_os_abstraction.h"
#include "fsl_tpm.h"
#include "TimersManager.h"
#include "genfsk_interface.h"

#include "gtof_debug.h"
#include "gtof_core.h"
#include "dtest_ctrl.h"

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Defines & Macros Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#define TOF_TX_INPUT_TRIGGER							(DTEST_TX_DIG_EN_PAGE) /* tx_dig_enable signal */
#define TOF_RX_INPUT_TRIGGER							(DTEST_AA_MACTH_PAGE) /* aa_sfd_match signal */

/* KW36/38 timestamp setup */
#if defined(RADIO_IS_GEN_3P5)
#define TS_ON_TPM2
#define TOF_TIMESTAMP_CLOCK_INSTANCE					(TPM2)
#define TOF_TPM_IRQ                                     (TPM2_IRQn)
#define TOF_TX_INPUT_CAPTURE_CHANNEL					(kTPM_Chnl_0)
#define TOF_TX_INPUT_CAPTURE_CHANNEL_FLAG				((uint32_t)kTPM_Chnl0Flag)
#define TOF_TX_INPUT_CAPTURE_CHANNEL_INTERRUPT          ((uint32_t)kTPM_Chnl0InterruptEnable)
#else
/* Legacy TPM for KW35, requires HW patch (removal of R) */
#define TOF_TIMESTAMP_CLOCK_INSTANCE					(TPM0)
#define TOF_TPM_IRQ                                     (TPM0_IRQn)
#define TOF_TX_INPUT_CAPTURE_CHANNEL					(kTPM_Chnl_0)
#define TOF_TX_INPUT_CAPTURE_CHANNEL_FLAG				((uint32_t)kTPM_Chnl0Flag)
#define TOF_TX_INPUT_CAPTURE_CHANNEL_INTERRUPT          ((uint32_t)kTPM_Chnl0InterruptEnable)
#endif

#define TOF_GET_CURRENT_AGC_INDEX() ((uint8_t)((XCVR_RX_DIG->AGC_STAT & XCVR_RX_DIG_AGC_STAT_CURR_AGC_IDX_MASK) >> XCVR_RX_DIG_AGC_STAT_CURR_AGC_IDX_SHIFT))

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                       Typedef Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Function Prototypes Section
///////////////////////////////////////////////////////////////////////////////////////////////////
void ToF_TPM_IRQHandler(void);

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Global Constants Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Static Constants Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Global Variables Section
///////////////////////////////////////////////////////////////////////////////////////////////////

/* Store some XCVR state */
tof_xcvr_data_t TofXcvrData;

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Static Variables Section
///////////////////////////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Functions Section
///////////////////////////////////////////////////////////////////////////////////////////////////

void Tof_TriggerEnableTx(void)
{
    TofXcvrData.txRxMode = TOF_TX_MODE;
    XCVR_DTEST_CTRL = XCVR_CTRL_DTEST_CTRL_DTEST_PAGE(TOF_TX_INPUT_TRIGGER) | XCVR_CTRL_DTEST_CTRL_DTEST_EN(1U);
}

void Tof_TriggerEnableRx(void)
{
    TofXcvrData.txRxMode = TOF_RX_MODE;
#if !defined(RADIO_IS_GEN_3P5)
    while((( XCVR_MISC->XCVR_STATUS & XCVR_CTRL_XCVR_STATUS_TSM_COUNT_MASK) >> XCVR_CTRL_XCVR_STATUS_TSM_COUNT_SHIFT ) != TofXcvrData.rxWarmupCount){};
#endif
    XCVR_DTEST_CTRL = XCVR_CTRL_DTEST_CTRL_DTEST_PAGE(TOF_RX_INPUT_TRIGGER) | XCVR_CTRL_DTEST_CTRL_DTEST_EN(1U);
}

void Tof_TriggerDisable(void)
{
    XCVR_DTEST_CTRL = 0;
}

/*
 * ===== Timestamp section
 */
void Tof_TimestampTimerInit(const gtofAppConfig_t *appConfig, toftimestampCb_t callback)
{
    tpm_config_t TpmConfig;

    /* Set IRQ callback */
    TofXcvrData.timestampHandlingCb = callback;
    /* Set IRQ Handler */
    OSA_InstallIntHandler((uint32_t)TOF_TPM_IRQ, ToF_TPM_IRQHandler);

#if defined(RADIO_IS_GEN_3P5)
    /* Setup source of RADIO_TOF_TS_TRIG  */

    /* tx_dig_enable is the trigger in Tx mode */
    XCVR_MISC->XCVR_CTRL |= XCVR_MISC_XCVR_CTRL_TOF_TX_SEL(0U);
    /* aa_fnd_to_ll is the trigger in Rx mode */
    XCVR_MISC->XCVR_CTRL |= XCVR_MISC_XCVR_CTRL_TOF_RX_SEL(0U);

    /* Connect RADIO_TOF_TS_TRIG signal to TPM2_CH0 (set as source) */
    SIM->SOPT4 |= SIM_SOPT4_TPM2CH0SRC(0x02U);

    /* TODO: workaround for XCVR HW bug TKT0515716.
     * coex clock needs to be active for tof_ts_trig to be propagated to SoC.
     * Should be removed when HW fix available.
     */
    XCVR_MISC->COEX_CTRL |= XCVR_MISC_COEX_CTRL_RF_NOT_ALLOWED_EN(0x2U);
#endif

    /* Select XTAL as the TPM clock source */
    CLOCK_SetTpmClock(2U);

    /* Initialize TPM input capture on rising edge */
    TPM_GetDefaultConfig(&TpmConfig);
    TPM_Init(TOF_TIMESTAMP_CLOCK_INSTANCE, &TpmConfig);
    TPM_SetupInputCapture(TOF_TIMESTAMP_CLOCK_INSTANCE, TOF_TX_INPUT_CAPTURE_CHANNEL, kTPM_RisingEdge);
    TPM_SetTimerPeriod(TOF_TIMESTAMP_CLOCK_INSTANCE,0xFFFF);

    TofXcvrData.timerReferenceClockHZ = CLOCK_GetFreq(kCLOCK_Osc0ErClk);
    (void)EnableIRQ(TOF_TPM_IRQ);

#if !defined(RADIO_IS_GEN_3P5)
    BOARD_InitTof();
#endif
    TMR_NotifyClkChanged();

    TofXcvrData.rxWarmupCount = (XCVR_TSM->END_OF_SEQ & XCVR_TSM_END_OF_SEQ_END_OF_RX_WU_MASK) >> XCVR_TSM_END_OF_SEQ_END_OF_RX_WU_SHIFT;

    /* Based on datarate, fractional should be shift by 2 or 3 bits (see TPM2_IRQHandler) */
    switch (appConfig->radioConfig.dataRate) {
    case gGenfskDR1Mbps:
        TofXcvrData.fracBitShift = 2;
        break;
#if defined(RADIO_IS_GEN_3P0)
    case gGenfskDR2Mbps:
        TofXcvrData.fracBitShift = 3;
        break;
#endif
    default:
        TofXcvrData.fracBitShift = 0;
        break;
    }

    Tof_TriggerDisable();
}

void Tof_TimestampTimerStart(void)
{
    TPM_EnableInterrupts(TOF_TIMESTAMP_CLOCK_INSTANCE, TOF_TX_INPUT_CAPTURE_CHANNEL_INTERRUPT);

	/* set period to the longest possible*/
	TPM_SetTimerPeriod(TOF_TIMESTAMP_CLOCK_INSTANCE,0xFFFF);

	/* reset the timer counter */
	TOF_TIMESTAMP_CLOCK_INSTANCE->CNT = 0;

	TPM_StartTimer(TOF_TIMESTAMP_CLOCK_INSTANCE, kTPM_SystemClock);
}

void Tof_TimestampTimerStop(void)
{
    /* Stop capture TPM, and disable TPM trigger */ 
    TPM_StopTimer(TOF_TIMESTAMP_CLOCK_INSTANCE);
    TPM_DisableInterrupts(TOF_TIMESTAMP_CLOCK_INSTANCE, TOF_TX_INPUT_CAPTURE_CHANNEL_INTERRUPT);

    TOF_TIMESTAMP_CLOCK_INSTANCE->CNT = 0;

    Tof_TriggerDisable();
}

void ToF_TPM_IRQHandler(void)
{
    uint32_t TimerFlags;

    TimerFlags = TPM_GetStatusFlags(TOF_TIMESTAMP_CLOCK_INSTANCE);
    TPM_ClearStatusFlags(TOF_TIMESTAMP_CLOCK_INSTANCE, TimerFlags);

    /* verify the input capture generated the interrupt */
    if ((TimerFlags & TOF_TX_INPUT_CAPTURE_CHANNEL_FLAG) != 0U)
    {
        uint32_t timestamp = TOF_TIMESTAMP_CLOCK_INSTANCE->CONTROLS[TOF_TX_INPUT_CAPTURE_CHANNEL].CnV;
        Tof_TriggerDisable();

#if defined(RADIO_IS_GEN_3P5)
        /* Radio 3.5 supports a specific timing feature to adjust timestamp trigered by AA_fnd. */
        GTOF_DEBUG_IO_TOGGLE(0);
        GTOF_DEBUG_IO_TOGGLE(0);

        if (TofXcvrData.txRxMode != TOF_TX_MODE)
        {
            uint32_t frac = (XCVR_2P4GHZ_PHY->STAT0 & XCVR_2P4GHZ_PHY_STAT0_FRAC_MASK) >> XCVR_2P4GHZ_PHY_STAT0_FRAC_SHIFT;

            /* The fractional value represents adjustment between true peak an AA_found signal. It is a signed Q6.5.
            * We need to substract this value from timestamp (positive frac means event occured earlier than timestamp).
            * - Check the sign bit and compute 2's complement if negative (obtain abs value)
            * - Right shift the value by 3 or 2 bits based on PHY clock (8MHz or 4MHz, i.e. 2Mbps or 1Mbps)
            *   This is to convert the 4MHz frac to 64MHz ticks.
            *   Ex for 4MHz: TimeStampAdjust = (FRAC/2^5)*(32MHz/4MHz) = FRAC/4
            */
            /* Check signed bit of Frac */
            if ((frac & 0x0020u) != 0u)
            {
                /* Frac is negative, takes 2's complement (abs value) */
                frac =  (0xFFFFFFFFu ^ frac) + 1u;
                frac &= 0x1Fu; /* Keep 5 LSB bits */
                frac = frac >> TofXcvrData.fracBitShift;
                timestamp += frac;
            }
            else
            {
                /* Frac is positive */
                frac = frac >> TofXcvrData.fracBitShift;
                timestamp -= frac;
            }
        }
#endif
        /* Timestamp is modulo 0xFFFF, wrapping is managed by the rest of the code */
        TofXcvrData.timestampHandlingCb(TofXcvrData.txRxMode,
                                        (uint16_t)timestamp,
                                        (uint16_t)TOF_GET_CURRENT_AGC_INDEX());
    }
}

void Tof_XcvrCustomConfiguration(bool isEnabled)
{
    static uint32_t LastDcValue;
    static uint32_t LastAgcStepConfig;

    if(isEnabled)
    {
        /* disable DC residual at all the AGC indexes but 26*/
        uint32_t NewValue;
        LastDcValue = XCVR_RX_DIG->DC_RESID_CTRL;
        NewValue = LastDcValue & ~XCVR_RX_DIG_DC_RESID_CTRL_DC_RESID_MIN_AGC_IDX_MASK;
        NewValue |= XCVR_RX_DIG_DC_RESID_CTRL_DC_RESID_MIN_AGC_IDX(26);
        XCVR_RX_DIG->DC_RESID_CTRL = NewValue;
        /* set AGC step to 1 for better accuracy */
        uint32_t AgcStepConfig;
        LastAgcStepConfig = XCVR_RX_DIG->AGC_CTRL_0;
        AgcStepConfig = LastAgcStepConfig;
        AgcStepConfig &= ~(XCVR_RX_DIG_AGC_CTRL_0_AGC_DOWN_BBA_STEP_SZ_MASK | XCVR_RX_DIG_AGC_CTRL_0_AGC_DOWN_LNA_STEP_SZ_MASK);
        AgcStepConfig |= XCVR_RX_DIG_AGC_CTRL_0_AGC_DOWN_BBA_STEP_SZ(1) | XCVR_RX_DIG_AGC_CTRL_0_AGC_DOWN_LNA_STEP_SZ(1);
        XCVR_RX_DIG->AGC_CTRL_0 = AgcStepConfig;
    }
    else
    {
        /* restore previous value */
        XCVR_RX_DIG->DC_RESID_CTRL = LastDcValue;
        XCVR_RX_DIG->AGC_CTRL_0 = LastAgcStepConfig;
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// EOF
///////////////////////////////////////////////////////////////////////////////////////////////////
